<?php

namespace App\Http\Controllers\Apis;

use App\Models\Post;
use App\Models\NewsLanguage;
use Illuminate\Http\Request;
use App\Models\NewsLanguageStatus;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Models\NewsLanguageSubscriber;

class NewsLanguageApiController extends Controller
{
    //<><><><><><> GET ALL NEWS LANGUAGES  <><><><><><>

    
    public function getNewsLanguages(Request $request)
    {
        $latestStatus = NewsLanguageStatus::latest('created_at')->first();
        if ($latestStatus && $latestStatus->status === 'active') {
            // Automatically get user ID from auth
            $userId = Auth::id();
           
            $followedLanguageIds = DB::table('news_languages_subscribers')
                ->where('user_id', $userId)
                ->pluck('news_language_id')
                ->toArray();
                
            $defaultLanguage = NewsLanguage::where('is_active', 1)->first();

            $news_languages = NewsLanguage::where('status','active')->get();
            $news_languages = $news_languages->map(function ($lang) use ($followedLanguageIds, $defaultLanguage) {
                $lang->image = $lang->image ? url('storage/' . $lang->image) : null;
                $lang->is_selected = in_array((int)$lang->id, array_map('intval', $followedLanguageIds)) || ($defaultLanguage && $lang->id == $defaultLanguage->id);
                $attributes = $lang->getAttributes();
                $lang->setRawAttributes($attributes);
                $lang->syncOriginal();
                return $lang;
            });
    
            return response()->json([
                'error' => false,
                'data' => $news_languages,
                'message' => 'News languages retrieved successfully',
            ]);
        }
        return response()->json([
            'error' => true,
            'data' => [],
            'default_language' => null,
            'message' => 'No active news language status found because the news language status is inactive',
        ]);
    }
    
    public function getPostsByNewsLanguage(Request $request)
    {
        $newsLanguageIds = $request->input('news_language_id');

        // Accept both array or string format
        if (!is_array($newsLanguageIds)) {
            $newsLanguageIds = explode(',', str_replace(['[', ']'], '', $newsLanguageIds));
        }

        $newsLanguageIds = array_filter($newsLanguageIds);

        if (empty($newsLanguageIds)) {
            return response()->json([
                'error' => true,
                'data' => [],
                'message' => 'One or more news language IDs are required as an array',
            ]);
        }

        // Check if language feature is active
        $isLanguageActive = NewsLanguageStatus::latest('created_at')->first();
        if (!$isLanguageActive || $isLanguageActive->status !== 'active') {
            return response()->json([
                'error' => true,
                'data' => [],
                'message' => 'News languages are currently inactive',
            ]);
        }

        // Validate IDs
        $existingIds = NewsLanguage::whereIn('id', $newsLanguageIds)->pluck('id')->toArray();
        $invalidIds = array_diff($newsLanguageIds, $existingIds);
        if (!empty($invalidIds)) {
            return response()->json([
                'error' => true,
                'data' => [],
                'message' => 'Invalid news language ID(s): ' . implode(', ', $invalidIds),
            ]);
        }

        // Get name and code of valid languages
        $languageData = NewsLanguage::whereIn('id', $newsLanguageIds)
            ->pluck('name', 'id')
            ->toArray();

        // Optional: add 'code' if needed
        $languageCodes = NewsLanguage::whereIn('id', $newsLanguageIds)
            ->pluck('code', 'id')
            ->toArray();

        // Get posts and attach language name/code
        $posts = Post::whereIn('news_language_id', array_keys($languageData))
            ->orderBy('publish_date', 'desc')
            ->get(['news_language_id'])
            ->map(function ($post) use ($languageData, $languageCodes) {
                return [
                    'id'   => $post->news_language_id,
                    'name' => $languageData[$post->news_language_id] ?? null,
                    'code' => $languageCodes[$post->news_language_id] ?? null, // optional
                ];
            })->unique('news_language_id')->values(); // Prevent duplicate language rows

        return response()->json([
            'error' => false,
            'data' => $posts,
            'message' => 'Posts retrieved successfully',
        ]);
    }
}
